package com.speed.print;

import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothDevice;
import android.content.Context;
import android.content.Intent;

import java.util.ArrayList;
import java.util.Set;
import java.util.Vector;

public class PrintManager {
    private static Context mContext;
    private ArrayList<String> mConnectedDeviceList;
    private BluetoothAdapter mBluetoothAdapter;
    private int id = 0;
    private ThreadPool mThreadPool;
    private PrintStateCallBack mPrintStateCallBack;
    /**
     * 使用打印机指令错误
     */
    private static final int PRINTER_COMMAND_ERROR = 0x008;
    public static final int BLUETOOTH_NOT_LINK = 0x009;

    public static final int REQUEST_ENABLE_BT = 0x010;


    private PrintManager() {
    }


    public static PrintManager getInstance(Context context) {
        mContext = context;
        return SingleTonHolder.INSTANCE;
    }

    private static class SingleTonHolder {
        private static final PrintManager INSTANCE = new PrintManager();
    }

    public boolean initBluetooth() {
        mBluetoothAdapter = BluetoothAdapter.getDefaultAdapter();
        if (mBluetoothAdapter == null) {
            Utils.toast(mContext, "当前设备不支持蓝牙");
        } else {
            if (!mBluetoothAdapter.isEnabled()) {
                if (mPrintStateCallBack != null) {
                    mPrintStateCallBack.onError(BLUETOOTH_NOT_LINK, "蓝牙功能未打开");
                }
            } else {
                if (!isPrintLink()) {
                    Utils.toast(mContext, "正在连接打印机");
                    return initPairedDevice();
                }
            }
        }
        return false;
    }


    /**
     * 设备配对
     */
    public boolean initPairedDevice() {
        mConnectedDeviceList = new ArrayList<>();
        Set<BluetoothDevice> pairedDevices = mBluetoothAdapter.getBondedDevices();
        if (pairedDevices.size() > 0) {
            for (BluetoothDevice device : pairedDevices) {
                if (device.getName() != null && device.getName().startsWith("Printer_")) {
                    mConnectedDeviceList.add(device.getAddress());
                }
            }
            if (mConnectedDeviceList.size() == 0) {
                return false;//无设备
            }
            //暂时只支持同时连接一个
            closePort();
            //初始化话DeviceConnFactoryManager
            new DeviceConnFactoryManager.Build()
                    .setId(id)
                    //设置连接方式
                    .setConnMethod(DeviceConnFactoryManager.CONN_METHOD.BLUETOOTH)
                    //设置连接的蓝牙mac地址
                    .setMacAddress(mConnectedDeviceList.get(0))
                    .setContext(mContext)
                    .build();
            //打开端口
            mThreadPool = ThreadPool.getInstantiation();
            mThreadPool.addSerialTask(new Runnable() {
                @Override
                public void run() {
                    DeviceConnFactoryManager.getDeviceConnFactoryManagers()[id].openPort();
                }
            });
            return true;
        } else {
            Utils.toast(mContext, "请先将打印机与手机进行蓝牙配对");
            return false;
        }
    }


    /**
     * 重新连接回收上次连接的对象，避免内存泄漏
     */
    private void closePort() {
        if (DeviceConnFactoryManager.getDeviceConnFactoryManagers()[id] != null && DeviceConnFactoryManager.getDeviceConnFactoryManagers()[id].mPort != null) {
            DeviceConnFactoryManager.getDeviceConnFactoryManagers()[id].cancel();
            DeviceConnFactoryManager.getDeviceConnFactoryManagers()[id].mPort.closePort();
            DeviceConnFactoryManager.getDeviceConnFactoryManagers()[id].mPort = null;
        }
    }


    /**
     * 判断打印机是否连接
     *
     * @return
     */
    public boolean isPrintLink() {
        boolean isLink;
        isLink = DeviceConnFactoryManager.getDeviceConnFactoryManagers()[id] != null && DeviceConnFactoryManager.getDeviceConnFactoryManagers()[id].getConnState();
        return isLink;
    }


    /**
     * 打印票据例子
     */
    public void btnReceiptPrint(final Vector<Byte> printContent) {
        if (!mBluetoothAdapter.isEnabled()) {
            if (mPrintStateCallBack != null) {
                mPrintStateCallBack.onError(BLUETOOTH_NOT_LINK, "蓝牙功能未打开");
            }
            return;
        }
        if (!isPrintLink()) {
            //Utils.toast(mContext, "正在连接打印机...");
            return;
        }
        mThreadPool = ThreadPool.getInstantiation();
        mThreadPool.addSerialTask(new Runnable() {
            @Override
            public void run() {
                if (DeviceConnFactoryManager.getDeviceConnFactoryManagers()[id].getCurrentPrinterCommand() == PrinterCommand.ESC) {
                    DeviceConnFactoryManager.getDeviceConnFactoryManagers()[id].sendDataImmediately(printContent);
                    if (mPrintStateCallBack != null) {
                        mPrintStateCallBack.onResponse("打印成功");
                    }
                } else {
                    if (mPrintStateCallBack != null) {
                        mPrintStateCallBack.onError(PRINTER_COMMAND_ERROR, "打印机命令错误");
                    }
                }
            }
        });
    }


    public void onDestory() {
        if (mBluetoothAdapter != null) {
            mBluetoothAdapter.cancelDiscovery();
        }
        mBluetoothAdapter = null;
        DeviceConnFactoryManager.closeAllPort();
        if (mThreadPool != null) {
            mThreadPool.stopThreadPool();
        }
        mThreadPool = null;
        mConnectedDeviceList = null;
    }

    public void setPrintStateCallBack(PrintStateCallBack printStateCallBack) {
        this.mPrintStateCallBack = printStateCallBack;
    }

    public interface PrintStateCallBack {
        void onResponse(String msg);

        void onError(int errorMsg, String msg);
    }
}
