package com.hc.printer.util

import android.annotation.SuppressLint
import android.app.Dialog
import android.content.Context
import android.content.Intent
import android.graphics.Bitmap
import android.graphics.Color
import android.provider.Settings
import android.text.TextUtils
import android.view.View
import android.widget.TableLayout
import android.widget.TableRow
import android.widget.TextView
import com.gprinter.command.EscCommand
import com.gprinter.command.LabelCommand
import com.hc.printer.R
import com.hc.printer.model.Goods
import com.hc.printer.model.MerchantOrder
import com.hc.printer.model.OrderInfo
import com.hc.printer.ui.BaseActivity
import com.hc.printer.ui.DoubleButtonDialog
import com.speed.print.Utils
import java.text.SimpleDateFormat
import java.util.*

/**
 * Created by danbo on 2020/6/29.
 */
object PrinterUtil {

    fun printOrder(context: Context, orderInfo: OrderInfo) {
        for (order in orderInfo.orders) {
            //判断订单是否打印过
            print(context, order)
        }
    }

    fun print(context: Context, order: MerchantOrder) {
        (context as BaseActivity<*>).let {
            it.mPrintManager?.apply {
                if (isPrintLink) {
                    btnReceiptPrint(getReceipt(context, order))
                } else {
                    initBluetooth()

                    Utils.toast(context, context.getString(R.string.str_cann_printer))

                    DoubleButtonDialog(
                        context,
                        null,
                        "未连接打印机",
                        "请先连接打印机后打印",
                        "取消",
                        "去连接",
                        object : DoubleButtonDialog.DialogClickListener {
                            override fun onClick(dialog: Dialog) {
                                it.startActivity(Intent(Settings.ACTION_BLUETOOTH_SETTINGS))
                            }

                            override fun onCancel(dialog: Dialog) {
                            }

                        }
                    ).show()
                }
            }
        }
    }

    /**
     * 票据打印测试页
     *
     * @return
     */
    @SuppressLint("SimpleDateFormat")
    private fun getReceipt(context: Context, order: MerchantOrder): Vector<Byte> {
        val esc = EscCommand()
        //初始化打印机
        esc.addInitializePrinter()

        // 设置打印居中
        esc.addSelectJustification(EscCommand.JUSTIFICATION.CENTER)

        //设置店铺文字大小
        esc.addSetCharcterSize(EscCommand.WIDTH_ZOOM.MUL_2, EscCommand.HEIGHT_ZOOM.MUL_2)
        // 店铺名称
        esc.addText(order.merchantName)
        //打印并换行
        esc.addPrintAndLineFeed()
        esc.addPrintAndLineFeed()

        //设置欢迎标语
        esc.addSetCharcterSize(EscCommand.WIDTH_ZOOM.MUL_1, EscCommand.HEIGHT_ZOOM.MUL_1)
        // 欢迎标语
        esc.addText("欢 迎 您 的 光 临")
        //打印并换行
        esc.addPrintAndLineFeed()

        // 设置打印左对齐
        esc.addSelectJustification(EscCommand.JUSTIFICATION.LEFT)

        //订单工号
        esc.addPrintAndLineFeed()
        esc.addText("工号：" + order.memberId)
        //订单编号
        esc.addPrintAndLineFeed()
        esc.addText("单号：" + order.orderSn)
        esc.addPrintAndLineFeed()

        //还原文字大小
        esc.addText("--------------------------------")
        esc.addPrintAndLineFeed()

        // 打印图片  光栅位图  384代表打印图片像素  0代表打印模式
        // 58mm打印机 可打印区域最大点数为 384 ，80mm 打印机 可打印区域最大点数为 576
        esc.addRastBitImage(getBitmap(context, order.goodsList), 384, 80)

        esc.addText("--------------------------------")
        esc.addPrintAndLineFeed()
        esc.addSetCharcterSize(EscCommand.WIDTH_ZOOM.MUL_1, EscCommand.HEIGHT_ZOOM.MUL_1)

        // 设置打印居右对齐
        esc.addSelectJustification(EscCommand.JUSTIFICATION.RIGHT)
        //打印fontB文字字体
        esc.addSetCharcterSize(EscCommand.WIDTH_ZOOM.MUL_1, EscCommand.HEIGHT_ZOOM.MUL_1)
        esc.addText("商品合计:${order.goodsList.size}件  ${order.totalPrice}元")
        esc.addPrintAndLineFeed()
        esc.addPrintAndLineFeed()

        // 设置打印左对齐
        esc.addSelectJustification(EscCommand.JUSTIFICATION.LEFT)

        //支付方式
        esc.addText("支付方式:")
        esc.addText(order.payType)
        esc.addPrintAndLineFeed()

        //实收
        esc.addText("实收:")
        esc.addText(order.totalPrice)
        esc.addPrintAndLineFeed()

        //找零
        esc.addText("找零:")
        esc.addText("0.00")
        esc.addPrintAndLineFeed()

        esc.addText(order.orderTime)

        esc.addPrintAndLineFeed()
        esc.addPrintAndLineFeed()

        esc.addSelectJustification(EscCommand.JUSTIFICATION.CENTER)
        // 结束标语
        esc.addText("谢谢惠顾！ 欢迎下次光临！")
        //打印并换行
        esc.addPrintAndLineFeed()

        //打印走纸n个单位
        esc.addPrintAndFeedLines(4.toByte())

        // 开钱箱
        esc.addGeneratePlus(LabelCommand.FOOT.F2, 255.toByte(), 255.toByte())
        //开启切刀
        esc.addCutPaper()
        //添加缓冲区打印完成查询
        val bytes = byteArrayOf(0x1D, 0x72, 0x01)
        //添加用户指令
        esc.addUserCommand(bytes)
        return esc.command
    }

    /**
     * 获取图片
     * @param context
     * @return
     */
    private fun getBitmap(context: Context, subOrderList: List<Goods>): Bitmap {
        val v = View.inflate(context, R.layout.pj, null)
        val tableLayout = v.findViewById(R.id.li) as TableLayout

        tableLayout.addView(
            ctv(
                context,
                "条码/品名",
                "数量",
                "单价"
            )
        )
        for (subOrder in subOrderList) {
            tableLayout.addView(
                ctv(
                    context,
                    "" + subOrder.name,
                    "" + subOrder.quantity,
                    "" + subOrder.price
                )
            )
        }

        return convertViewToBitmap(v)
    }

    private fun ctv(context: Context, name: String, n: String, t: String?): TableRow {
        val tb = TableRow(context)
        tb.layoutParams = TableLayout.LayoutParams(
            TableLayout.LayoutParams.WRAP_CONTENT,
            TableLayout.LayoutParams.WRAP_CONTENT
        )
        val tv1 = TextView(context)
        val layoutParams = TableRow.LayoutParams(
            TableRow.LayoutParams.WRAP_CONTENT,
            TableRow.LayoutParams.WRAP_CONTENT
        )
        layoutParams.topMargin = ScreenUtils.dip2px(4f)
        layoutParams.rightMargin = ScreenUtils.dip2px(20f)
        tv1.layoutParams = layoutParams

        tv1.text = name
        tv1.maxEms = 6
        tv1.maxLines = 2
        tv1.ellipsize = TextUtils.TruncateAt.END
        tv1.setTextColor(Color.BLACK)
        tv1.textSize = 10f
        tb.addView(tv1)

//        val tv2 = TextView(context)
//
//        tv2.layoutParams = layoutParams
//
//        tv2.text = k.toString()
//        tv2.setTextColor(Color.BLACK)
//        tv2.textSize = 10f
//        tb.addView(tv2)

        val tv3 = TextView(context)
        tv3.layoutParams = layoutParams

        tv3.text = n.toString()
        tv3.setTextColor(Color.BLACK)
        tv3.textSize = 10f
        tb.addView(tv3)

        val tv4 = TextView(context)
        tv4.layoutParams = layoutParams

        tv4.text = t
        tv4.setTextColor(Color.BLACK)
        tv4.textSize = 10f
        tb.addView(tv4)
        return tb
    }

    /**
     * mxl转bitmap图片
     * @param view
     * @return
     */
    private fun convertViewToBitmap(view: View): Bitmap {
        view.measure(
            View.MeasureSpec.makeMeasureSpec(0, View.MeasureSpec.UNSPECIFIED),
            View.MeasureSpec.makeMeasureSpec(0, View.MeasureSpec.UNSPECIFIED)
        )
        view.layout(0, 0, view.measuredWidth, view.measuredHeight)
        view.buildDrawingCache()
        return view.drawingCache
    }
}