package com.hc.printer.ui

import android.app.Activity
import android.bluetooth.BluetoothAdapter
import android.content.Intent
import android.graphics.Color
import android.os.Bundle
import android.view.View
import android.view.View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN
import android.view.View.SYSTEM_UI_FLAG_LIGHT_STATUS_BAR
import androidx.appcompat.app.AppCompatActivity
import androidx.databinding.DataBindingUtil
import androidx.databinding.ViewDataBinding
import androidx.lifecycle.ViewModelProviders
import androidx.lifecycle.ViewModelStore
import com.hc.printer.MainActivity
import com.hc.printer.R
import com.hc.printer.BR
import com.hc.printer.util.Utils
import com.hc.printer.util.VMUtil
import com.hc.printer.viewmodel.BaseViewModel
import com.speed.print.PrintManager

/**
 * Created by danbo on 2019/11/07.
 */
abstract class BaseActivity<VM : BaseViewModel> : AppCompatActivity(), BaseView {
    open lateinit var mBinding: ViewDataBinding

    open var mViewModel: VM? = null

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        window.decorView.systemUiVisibility = (SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN
                or View.SYSTEM_UI_FLAG_LAYOUT_STABLE
                or SYSTEM_UI_FLAG_LIGHT_STATUS_BAR)

        //状态栏透明
        window.statusBarColor = Color.TRANSPARENT
        window.navigationBarColor = resources.getColor(R.color.navigationBarColor)

        Utils.init(this)

        val layoutId = getLayoutId()
        if (layoutId == 0) {
            return
        }

        // Obtain ViewModel from ViewModelProviders
        val vm = VMUtil.getVM<VM>(this)
        vm?.let {
            mViewModel = ViewModelProviders.of(this).get(vm::class.java)
        }
        //bind context
        mViewModel!!.context = this

        // Obtain binding
        mBinding =
            DataBindingUtil.setContentView(this, layoutId)

        //bind ViewModel
        mBinding.setVariable(getBR(), mViewModel)

        // LiveData needs the lifecycle owner
        mBinding.lifecycleOwner = this

        //首页和订单详情页需要打印机
        if (this is MainActivity) {
            initPrintManager()
        }

        initView()
    }

    var mPrintManager: PrintManager? = null

    fun initPrintManager(): Boolean {
        com.speed.print.Utils.toast(this@BaseActivity, "初始化打印机")

        mPrintManager = PrintManager.getInstance(this)

        mPrintManager?.setPrintStateCallBack(object :
            PrintManager.PrintStateCallBack {
            override fun onResponse(msg: String) {
                println("--Printer--$msg")
            }

            override fun onError(errorMsg: Int, msg: String) {
                println("--Printer--$msg")
                if (errorMsg == PrintManager.BLUETOOTH_NOT_LINK) {
                    val enableIntent = Intent(BluetoothAdapter.ACTION_REQUEST_ENABLE)
                    startActivityForResult(enableIntent, PrintManager.REQUEST_ENABLE_BT)
                }
            }
        })

        return mPrintManager!!.initBluetooth()
    }

    override fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent?) {
        super.onActivityResult(requestCode, resultCode, data)
        if (requestCode == PrintManager.REQUEST_ENABLE_BT) {
            if (resultCode == Activity.RESULT_OK) {
                mPrintManager?.initPairedDevice()
            } else {
                com.speed.print.Utils.toast(
                    this@BaseActivity,
                    getString(R.string.bluetooth_is_not_enabled)
                )
            }
        }
    }

    override fun getBR(): Int {
        return BR.viewModel
    }

}
